%% Simulate overcapacity, Feast/famine
% Introduce an extra sector which contains proteins that are not necessary
% to obtain the simulaed flux profile, and try to allocate as many proteins
% as possible into this sector
clear all; close all; clc

phi = cell2mat(struct2cell(load('ProteomeOptimization_FF_1_20.mat', 'f_opt')));
phi_org = phi;

% Add an extra sector
names = {'Upt', 'UGlc', 'LGlc', 'Ferm', 'ESnk', 'Resp', 'Treh', 'Grwt', 'Struc', 'Extra'};
phi(10) = 0;

% Define conditions
D = 0.1;                                % 1/h

% Chemostat experiment
c0 = [1e-6 100 0 0 6.36 0 33.12 0 0 0 0];  % mM
chsttime = 100;                          % h

% Simulate chemostat to get initial conditions feas/famine
options = odeset('NonNegative', 1:numel(c0), 'AbsTol', 1e-4);
[t_ref_ch, c_ref_ch] = ode15s(@(t,c) protmdl(t,c,phi_org,D,1,1), [0:chsttime], c0, options);
c0 = c_ref_ch(end,:);

% Feast/famine experiment
ti = 0; tf = 25+800/3600;               % h

tcycle = 400/3600;                      % h
tfeed = (400/20)/3600;    % h   
numcycles = tf/tcycle;  

% Simulate feast/famine experiment per cycle
t_ref = []; c_ref = []; v_ref = [];

% Stress response penalty for first cycle
f_mtn_req = 0.0203/(1+(c0(1)/0.02)^21)+0.0618;
delta_f = phi(9) - f_mtn_req;
if delta_f < 0
    penalty = abs(delta_f)*100*0.0103;
else
    penalty = 0;
end

% Don't switch off fermentation in the first cycle
fermsw = 1;

for i = 1:numcycles
    t_ref_ffc = []; c_ref_ffc = [];
    
    % Set time interval of i-th cycle
    t_start = (i-1) * tcycle;
    t_end = i * tcycle;
    
    % Simulate i-th cycle
    [t_ref_ffc, c_ref_ffc] = ode15s(@(t,c) protmdl_ff(t,c,phi_org,D,tcycle,tfeed,penalty,fermsw), [t_start t_end], c0, options);
    c0 = c_ref_ffc(end,:);
    
    % Stress response penalty based on the maximum and minimum residual
    % glucose concentration for the next cycle
    cav = (max(c_ref_ffc(:,1)) + min(c_ref_ffc(:,1))) / 2;
    f_mtn_req = 0.0203/(1+(cav/0.02)^21)+0.0618;
    delta_f = phi(9) - f_mtn_req;
    if delta_f < 0
        penalty = abs(delta_f)*100*0.0103;
    else
        penalty = 0;
    end
    
    % Switch off fermentation if the maximum glucose concentration in the
    % previous cycle is too low
    if max(c_ref_ffc(:,1)) < 0.085
        fermsw = 0;
    else
        fermsw = 1;
    end
end

% Simulate last cycle for reference flux profile
[t_ref, c_ref] = ode15s(@(t,c) protmdl_ff(t,c,phi,D,tcycle,tfeed,penalty,fermsw), [t_start t_end], c0, options);

for i = 1:length(t_ref)
    [~,v_ref(i,:)] = protmdl_ff(t_ref(i),c_ref(i,:)',phi,D,tcycle,tfeed,penalty,fermsw);
end

%% Overcapacity simulation
f_seed = phi;

% First move proteins that are in overcapacity based on maximum sector
% sizes
f_upt_max = 0.016;
f_mit_max = 0.12;

if f_seed(1) > f_upt_max
    f_seed(10) = f_seed(10) + (f_seed(1) - f_upt_max);
    f_seed(1) = f_upt_max;
end

if f_seed(6) > f_mit_max
    f_seed(10) = f_seed(10) + (f_seed(6) - f_mit_max);
    f_seed(6) = f_mit_max;
end

% Reduce the structure sector to the minimal required value at this
% residual glucose concentration. The sector is not adapted if the 
% simulated cells already receive a penalty.
cav = (max(c_ref_ffc(:,1)) + min(c_ref_ffc(:,1))) / 2;
f_mtn_req = 0.0203/(1+(cav/0.02)^21)+0.0618;
if f_mtn_req < f_seed(9)
    f_seed(10) = f_seed(10) + (f_seed(9) - f_mtn_req);
    f_seed(9)  = f_mtn_req;
end

% Update optimized proteome
f_opt = f_seed;
f_opt = phi;
% Start with the biggest sector
f_sort = f_opt(1:8);
[f_sort, sortindex] = sort(f_sort,'descend');

for s = sortindex
    % Reset switch
    stop = 0;
    
    while stop == 0
        % Decrease the sector size by one percent
        f_seed     = f_opt;
        f_seed(10) = f_opt(10) + 0.01*f_opt(s);
        f_seed(s)  = f_opt(s) * 0.99;
        
        % Chemostat experiment
        c0 = [1e-6 100 0 0 6.36 0 33.12 0 0 0 0];  % mM

        % Simulate chemostat to get initial conditions feas/famine
        options = odeset('NonNegative', 1:numel(c0), 'AbsTol', 1e-4);
        [t_ch, c_ch] = ode15s(@(t,c) protmdl(t,c,f_opt,D,1,1), [0:chsttime], c0, options);
        c0 = c_ch(end,:);

        % Simulate feast/famine experiment per cycle
        t = []; c = []; v = [];

        % Stress response penalty for first cycle
        f_mtn_req = 0.0203/(1+(c0(1)/0.02)^21)+0.0618;
        delta_f = f_opt(9) - f_mtn_req;
        if delta_f < 0
            penalty = abs(delta_f)*100*0.0103;
        else
            penalty = 0;
        end

        % Don't switch off fermentation in the first cycle
        fermsw = 1;

        for i = 1:numcycles
            t_ffc = []; c_ffc = [];

            % Set time interval of i-th cycle
            t_start = (i-1) * tcycle;
            t_end = i * tcycle;

            % Simulate i-th cycle
            [t_ffc, c_ffc] = ode15s(@(t,c) protmdl_ff(t,c,f_seed,D,tcycle,tfeed,penalty,fermsw), [t_start t_end], c0, options);
            c0 = c_ffc(end,:);

            % Stress response penalty based on the maximum and minimum residual
            % glucose concentration for the next cycle
            cav = (max(c_ffc(:,1)) + min(c_ffc(:,1))) / 2;
            f_mtn_req = 0.0203/(1+(cav/0.02)^21)+0.0618;
            delta_f = f_seed(9) - f_mtn_req;
            if delta_f < 0
                penalty = abs(delta_f)*100*0.0103;
            else
                penalty = 0;
            end

            % Switch off fermentation if the maximum glucose concentration in the
            % previous cycle is too low
            if max(c_ffc(:,1)) < 0.085
                fermsw = 0;
            else
                fermsw = 1;
            end
        end

        % Simulate last cycle for reference flux profile
        [t, c] = ode15s(@(t,c) protmdl_ff(t,c,f_seed,D,tcycle,tfeed,penalty,fermsw), [t_start t_end], c0, options);

        for i = 1:length(t)
            [~,v(i,:)] = protmdl_ff(t(i),c(i,:)',f_seed,D,tcycle,tfeed,penalty,fermsw);
        end
        
        % Check for imbalanced states
        if max(c(:,7)) < 1
            % If phosphate is never high, it must be imbalanced
            unb = 1;
            disp('a')
        elseif max(c(:,8)) > 20
            % If intermediates reach high concentrations, it must be
            % imbalanced
            unb = 1;
            disp('b')
        elseif max(c(:,9)) > 20
            unb = 1;
            disp('c')
        elseif max(c(:,10)) > 20
            unb = 1;
            disp('d')
        elseif isreal(c) == 0
            unb = 1;
            disp('e')
        else
            unb = 0;
        end
        
        cs_t_avg = trapz(t,c(:,1).*t)/trapz(t,t);
        cs_t_avg_ref = trapz(t_ref,c_ref(:,1).*t_ref)/trapz(t_ref,t_ref);
        cs_ratio = cs_t_avg/cs_t_avg_ref;
        
        cx_dif = (c(1,2)-c(end,2))/c(1,2);
        cx_dif_ref = (c_ref(1,2)-c_ref(end,2))/c_ref(1,2);
        cx_ratio = cx_dif/cx_dif_ref;
                
        if unb == 1 || cs_ratio > 1.05 || cx_ratio < 0.95 || f_seed(s) < 0
            disp(['UPDATE, ', names{s}, ' = ', num2str(f_opt(s))])
            stop = 1;
        else
            f_opt = f_seed;
            disp([names{s}, ' = ', num2str(f_seed(s))])
            stop = 0;
        end
    end
end

% Chemostat experiment
c0 = [1e-6 100 0 0 6.36 0 33.12 0 0 0 0];  % mM

% Simulate chemostat to get initial conditions feas/famine
options = odeset('NonNegative', 1:numel(c0), 'AbsTol', 1e-4);
[t_ch, c_ch] = ode15s(@(t,c) protmdl(t,c,f_opt,D,1,1), [0:chsttime], c0, options);
c0 = c_ch(end,:);

% Simulate feast/famine experiment per cycle
t = []; c = []; v = [];

% Stress response penalty for first cycle
f_mtn_req = 0.0203/(1+(c0(1)/0.02)^21)+0.0618;
delta_f = f_opt(9) - f_mtn_req;
if delta_f < 0
    penalty = abs(delta_f)*100*0.0103;
else
    penalty = 0;
end

% Don't switch off fermentation in the first cycle
fermsw = 1;

for i = 1:numcycles
    t_ffc = []; c_ffc = [];

    % Set time interval of i-th cycle
    t_start = (i-1) * tcycle;
    t_end = i * tcycle;

    % Simulate i-th cycle
    [t_ffc, c_ffc] = ode15s(@(t,c) protmdl_ff(t,c,f_opt,D,tcycle,tfeed,penalty,fermsw), [t_start t_end], c0, options);
    c0 = c_ffc(end,:);

    % Stress response penalty based on the maximum and minimum residual
    % glucose concentration for the next cycle
    cav = (max(c_ffc(:,1)) + min(c_ffc(:,1))) / 2;
    f_mtn_req = 0.0203/(1+(cav/0.02)^21)+0.0618;
    delta_f = f_seed(9) - f_mtn_req;
    if delta_f < 0
        penalty = abs(delta_f)*100*0.0103;
    else
        penalty = 0;
    end

    % Switch off fermentation if the maximum glucose concentration in the
    % previous cycle is too low
    if max(c_ffc(:,1)) < 0.085
        fermsw = 0;
    else
        fermsw = 1;
    end
end

% Simulate last cycle for reference flux profile
[t, c] = ode15s(@(t,c) protmdl_ff(t,c,f_opt,D,tcycle,tfeed,penalty,fermsw), [t_start t_end], c0, options);

for i = 1:length(t)
    [~,v(i,:)] = protmdl_ff(t(i),c(i,:)',f_opt,D,tcycle,tfeed,penalty,fermsw);
end

%% Plot results

vnames = {'Upt', 'UGlc', 'LGlc', 'Ferm', 'ESnk', 'Resp, TCA', 'Resp, NDE', 'TrSn', 'TrDg', 'Grwt', 'Mtn'};
figure(1)
for i = 1:length(vnames)
    subplot(3,4,i)
    plot((t_ref-t_ref(1)).*3600, v_ref(:,i), 'LineWidth', 1.2, 'Color', autumn(1)); hold on
    plot((t-t(1)).*3600, v(:,i), 'LineWidth', 1.2, 'Color', winter(1)); hold on
    xlabel('Time (s)')
    ylabel('Flux (mol/Cmol/h)')
    title(vnames{i})
    xlim([0 tcycle*3600])
end

% Concentrations
figure(2)
cnames = {'Glucose', 'Biomass', 'Ethanol', 'Glycerol', 'ATP', 'NADH', 'Pi', 'G6P', 'FBP','Pyr','Trehalose'};
for i = 1:length(cnames)
    subplot(3,4,i)
    plot((t_ref-t_ref(1)).*3600, c_ref(:,i), 'LineWidth', 1.2, 'Color', autumn(1)); hold on
    plot((t-t(1)).*3600, c(:,i), 'LineWidth', 1.2, 'Color', winter(1)); hold on
    xlabel('Time (s)')
    ylabel('Concentration (mM)')
    title(cnames{i})
    xlim([0 tcycle*3600])
end
%%
save('ProteomeOptimization_FF_X_opt')